package ActivePerl::PPM::Profile;

use strict;

sub new {
    my($class, $profile_xml) = @_;

    require XML::Simple;
    my $self = XML::Simple::XMLin($profile_xml, ForceArray => [qw(REPOSITORY SOFTPKG)]);
    $self->{REPOSITORY} ||= [];
    $self->{SOFTPKG} ||= [];

    _downcase($self);
    if (my $ap = $self->{activeperl}) {
        for (values %$ap) {
            utf8::downgrade($_);
        }
    }

    bless $self, $class;
    return $self;
}

sub activeperl_version {
    my $self = shift;
    return $self->{activeperl}{version};
}

sub perl_version {
    my $self = shift;
    return $self->{activeperl}{perl_version};
}

sub ppm_version {
    my $self = shift;
    return $self->{activeperl}{ppm_version};
}

sub repositories {
    my $self = shift;
    my @repos = @{$self->{repository}};
    for (@repos) {
        $_->{enabled} = 1 unless exists $_->{enabled};
    }
    return @repos;
}

sub packages {
    my $self = shift;
    return @{$self->{softpkg}};
}

sub _downcase {
    my $o = shift;
    if (ref($o) eq "HASH") {
        for my $k (keys %$o) {
            my $lc_k = lc($k);
            $o->{$lc_k} = delete $o->{$k} if $lc_k ne $k;
            _downcase($o->{$lc_k});
        }
    }
    elsif (ref($o) eq "ARRAY") {
        _downcase($_) for @$o;
    }
}

1;

__END__

=head1 NAME

ActivePerl::PPM::Profile - Access to profile data

=head1 SYNOPSIS

  my $profile = ActivePerl::PPM::Profile->new(<<EOT);
  <PPMPROFILE>
    <ACTIVEPERL VERSION="1004" PERL_VERSION="5.10.0" PPM_VERSION="4.04"/>
    <REPOSITORY NAME="ActiveState Package Repository"
       HREF="http://ppm4.activestate.com/darwin/5.10/1004/package.xml"/>
    <SOFTPKG NAME="Foo-Bar" VERSION="1.0"/>
    <SOFTPKG NAME="HTML-Parser" VERSION="3.65"/>
  </PPMPROFILE>
  EOT

=head1 DESCRIPTION

The C<ActivePerl::PPM::Profile> object parse and provide access to
XML profile dumps as generated by the B<ppm profile save> command.

The following methods are provided:

=over

=item $profile = ActivePerl::PPM::Profile->new( $xml )

=item $profile = ActivePerl::PPM::Profile->new( $filename )

This will parse the given XML document and create the object that
wraps the data.  The method croaks if it can't parse the document or
can't open the file.

=item $profile->activeperl_version

This returns the build number of the ActivePerl used to create the
profile.  Example "825".

=item $profile->perl_version

This returns the perl version number of the ActivePerl used to create
the profile.  Example "5.8.9".

=item $profile->ppm_version

This return the version number of PPM that generated the profile.
Example "4.05".

=item $profile->repositories

This returns the list of repositories configured.  Each entry is a
hash with the keys C<name>, C<href> and C<enabled>.  In scalar context
returns the number of repositories.

=item $profile->packages

This returns the list of packages installed.  Each entry is a hash
with the keys C<name> and C<version>.  In scalar context returns the
number of packages.

=back

=head1 SEE ALSO

L<ppm>

